<?php
/* --------------------------------------------------------------
  ConfigSettings.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit;

/**
 * Class ConfigSettings
 *
 * @package StyleEdit
 */
class ConfigSettings
{
	/**
	 * @var string $currentTemplate The name of the active template.
	 */
	private $currentTemplate;
	
	/**
	 * @var string $templatesDirectory Absolute path to the current template directory where the style configuration
	 *                                 files are stored.
	 */
	private $templatesDirectory;
	
	/**
	 * @var string $stylesDirectory Absolute path to the templates directory inside StyleEdit3.
	 */
	private $stylesDirectory;
	
	/**
	 * @var string $originalThemeStylesDirectory Absolute path to the original styles directory inside themes.
	 */
	private $originalThemeStylesDirectory;
	
	/**
	 * @var string $boilerplatesDirectory Absolute path to the boilerplates directory of the current template.
	 */
	private $boilerplatesDirectory;
	
	/**
	 * @var array $additionalBoxesDirectories Absolute path to the additional boxes directories of the current template.
	 */
	private $additionalBoxesDirectories;
	
	
	/**
	 * ConfigSettings constructor.
	 *
	 * @param string $p_currentTemplate
	 *
	 * @throws \InvalidArgumentException if given template name is not a string or empty or resulting directories are
	 *                                   not existing and writable.
	 */
	public function __construct($p_currentTemplate)
	{
		if(!is_string($p_currentTemplate) || empty($p_currentTemplate))
		{
			throw new \InvalidArgumentException('Invalid $p_currentTemplate argument value (string expected): '
			                                    . print_r($p_currentTemplate, true));
		}
		
		$this->templatesDirectory = realpath(dirname(__DIR__) . '/templates') . '/';
		
		$currentTemplate   = str_replace('..', '', $p_currentTemplate);
		$templateDirectory = StyleThemeControl::getStylesDirectoryPath($currentTemplate);
		if(!$this->_isDir($templateDirectory))
		{
			throw new \InvalidArgumentException("Template directory '$templateDirectory' does not exist");
		}
		if(!$this->_isDirWritable($templateDirectory))
		{
			throw new \InvalidArgumentException("Template directory '$templateDirectory' is not writable");
		}
		
		$this->currentTemplate              = $currentTemplate;
		$this->stylesDirectory              = $templateDirectory;
		$this->originalThemeStylesDirectory = StyleThemeControl::getOriginalThemeStylesDirectoryPath($currentTemplate);
		
		$boilerplatesDirectory = $this->stylesDirectory . 'boilerplates/';
		if(!$this->_isDir($boilerplatesDirectory))
		{
			throw new \InvalidArgumentException("Boilerplates directory '$boilerplatesDirectory' does not exist");
		}
		
		$this->boilerplatesDirectory = $boilerplatesDirectory;
		
		$additionalBoxesDirectory = $this->stylesDirectory . 'boxes/';
		if(!$this->_isDir($additionalBoxesDirectory))
		{
			throw new \InvalidArgumentException("Additional boxes directory '$additionalBoxesDirectory' does not exist");
		}
		
		$cacheFile = \StyleEditConfig::getCacheDirectoryPath() . 'StyleEdit3_ConfigSettings_boxesDir.json';
		if(file_exists($cacheFile))
		{
			$boxesDir = json_decode(file_get_contents($cacheFile), true);
		}
		else
		{
			$boxesDir   = [];
			$boxesDir[] = $additionalBoxesDirectory;
			
			if(\StyleEditConfig::getAdditionalBoxesBasePath() !== null)
			{
				$dirIterator = new \RecursiveDirectoryIterator(\StyleEditConfig::getAdditionalBoxesBasePath());
				$iterator    = new \RecursiveIteratorIterator($dirIterator, \RecursiveIteratorIterator::SELF_FIRST);
				$iterations = 0;
				foreach($iterator as $file)
				{
					$boxesAll         = \StyleEditConfig::getAdditionalBoxesDirectoryPath();
					$boxesTemplate    = \StyleEditConfig::getAdditionalBoxesDirectoryPath($currentTemplate);
					$hasBoxesAll      = stripos($file->getPathname(), $boxesAll);
					$hasBoxesTemplate = stripos($file->getPathname(), $boxesTemplate);
					
					if($hasBoxesAll !== false || $hasBoxesTemplate !== false)
					{
						if($hasBoxesAll !== false)
						{
							$pos = $hasBoxesAll + strlen($boxesAll);
						}
						else
						{
							$pos = $hasBoxesTemplate + strlen($boxesTemplate);
						}
						
						$boxesDir[] = substr($file->getPathname(), 0, $pos);
					}
					$iterations++;
				}
			}
			file_put_contents($cacheFile, json_encode($boxesDir));
		}
		
		$this->additionalBoxesDirectories = array_unique($boxesDir);
	}
	
	
	/**
	 * Returns the name of the current template.
	 *
	 * @return string
	 */
	public function getCurrentTemplate()
	{
		return $this->currentTemplate;
	}
	
	
	/**
	 * Returns the absolute path to the templates directory inside StyleEdit3.
	 *
	 * @param bool $p_forOriginalTheme
	 *
	 * @return string
	 */
	public function getStylesDirectory($p_forOriginalTheme = false)
	{
		return $p_forOriginalTheme ? $this->originalThemeStylesDirectory : $this->stylesDirectory;
	}
	
	
	/**
	 * Returns the absolute path to the boilerplates directory of the current template.
	 *
	 * @return string
	 */
	public function getBoilerplatesDirectory()
	{
		return $this->boilerplatesDirectory;
	}
	
	
	/**
	 * Returns the absolute path to the current template directory where the style configuration files are stored.
	 *
	 * @return string
	 */
	public function getTemplatesDirectory()
	{
		return $this->templatesDirectory;
	}
	
	
	/**
	 * Returns the absolute path to the additional boxes directory of the current template.
	 *
	 * @return array
	 */
	public function getAdditionalBoxesDirectories()
	{
		return $this->additionalBoxesDirectories;
	}
	
	
	/**
	 * Checks if the given path exists in the filesystem.
	 *
	 * @param string $p_path A path to a directory.
	 *
	 * @return bool
	 */
	private function _isDir($p_path)
	{
		if(!is_dir($p_path))
		{
			return false;
		}
		
		return true;
	}
	
	
	/**
	 * Checks if the given path is writable.
	 *
	 * @param string $p_path A path to a directory.
	 *
	 * @return bool
	 */
	private function _isDirWritable($p_path)
	{
		if(!is_writable($p_path))
		{
			return false;
		}
		
		return true;
	}
}
